/**
 * Worker process for reading weight from an industrial scale via Modbus RTU
 * Based on leer_peso.js - communicates with main process via IPC
 */

const ModbusRTU = require("modbus-serial");

// Configuration from environment variables
const PORT = process.env.SCALE_PORT || "COM6";
const BAUD = parseInt(process.env.SCALE_BAUD || "9600", 10);
const SLAVE_ID = parseInt(process.env.SCALE_SLAVE_ID || "1", 10);
const START_ADDR = parseInt(process.env.SCALE_START_ADDR || "0", 10);
const POLL_MS = parseInt(process.env.SCALE_POLL_MS || "500", 10);
const WORD_ORDER = (process.env.SCALE_WORD_ORDER || "BA").toUpperCase();
const REGISTER_KIND = (process.env.SCALE_REGISTERS || "holding").toLowerCase();
const SCALE_DIVISOR = parseFloat(process.env.SCALE_DIV || "10000");
const DEBUG = process.env.SCALE_DEBUG === "1";

const client = new ModbusRTU();
let pollTimer = null;
let isConnected = false;
let isExiting = false;

function log(message) {
  if (DEBUG) {
    console.log("[ScaleWorker]", message);
  }
  sendMessage({ type: "log", message });
}

function sendMessage(message) {
  if (process.send && typeof process.send === "function") {
    try {
      process.send(message);
    } catch (error) {
      console.error("[ScaleWorker] Failed to send message:", error);
    }
  }
}

function sendUpdate(payload) {
  sendMessage({ type: "update", payload });
}

function sendError(error) {
  sendMessage({ type: "error", error: String(error) });
}

function toSigned32(hi, lo) {
  const unsigned = ((hi << 16) >>> 0) + (lo & 0xffff);
  return unsigned > 0x7fffffff ? unsigned - 0x100000000 : unsigned;
}

async function readTwoRegisters() {
  const method =
    REGISTER_KIND === "input"
      ? client.readInputRegisters.bind(client)
      : client.readHoldingRegisters.bind(client);
  const { data } = await method(START_ADDR, 2);
  const [r0, r1] = data;
  const [hi, lo] = WORD_ORDER === "BA" ? [r1, r0] : [r0, r1];
  return toSigned32(hi, lo);
}

async function connect() {
  if (isExiting) return false;
  
  try {
    log(`Connecting to scale on ${PORT} at ${BAUD} baud...`);
    
    await client.connectRTUBuffered(PORT, {
      baudRate: BAUD,
      dataBits: 8,
      stopBits: 1,
      parity: "none",
    });
    
    client.setID(SLAVE_ID);
    client.setTimeout(2000);
    
    isConnected = true;
    log(`Connected to scale on ${PORT}`);
    
    sendUpdate({
      ready: true,
      error: null,
      port: PORT,
    });
    
    return true;
  } catch (error) {
    isConnected = false;
    log(`Connection failed: ${error.message}`);
    sendUpdate({
      ready: false,
      error: `No se pudo conectar al puerto ${PORT}: ${error.message}`,
      port: PORT,
    });
    return false;
  }
}

async function readWeight() {
  if (!isConnected || isExiting) return;
  
  try {
    const raw = await readTwoRegisters();
    const kg = raw / SCALE_DIVISOR;
    // Round to 1 decimal place using toFixed to avoid floating point precision issues
    const kg1 = Number(kg.toFixed(1));
    
    sendUpdate({
      weight: kg1,
      rawWeight: raw,
      ready: true,
      error: null,
      updatedAt: Date.now(),
    });
    
    if (DEBUG) {
      log(`Weight: ${kg1.toFixed(1)} kg (raw: ${raw})`);
    }
  } catch (error) {
    log(`Read error: ${error.message}`);
    
    // Check if it's a connection error
    if (
      error.message.includes("Port Not Open") ||
      error.message.includes("disconnected") ||
      error.message.includes("ECONNRESET")
    ) {
      isConnected = false;
      sendUpdate({
        ready: false,
        error: `Conexión perdida: ${error.message}`,
      });
      
      // Stop polling and try to reconnect
      stopPolling();
      scheduleReconnect();
    } else {
      sendUpdate({
        error: `Error de lectura: ${error.message}`,
      });
    }
  }
}

function startPolling() {
  if (pollTimer) return;
  
  pollTimer = setInterval(() => {
    readWeight();
  }, POLL_MS);
  
  // Initial read
  readWeight();
}

function stopPolling() {
  if (pollTimer) {
    clearInterval(pollTimer);
    pollTimer = null;
  }
}

let reconnectTimer = null;

function scheduleReconnect() {
  if (isExiting || reconnectTimer) return;
  
  reconnectTimer = setTimeout(async () => {
    reconnectTimer = null;
    
    if (isExiting) return;
    
    const connected = await connect();
    if (connected) {
      startPolling();
    } else {
      scheduleReconnect();
    }
  }, 3000);
}

function cleanup() {
  isExiting = true;
  stopPolling();
  
  if (reconnectTimer) {
    clearTimeout(reconnectTimer);
    reconnectTimer = null;
  }
  
  try {
    client.close(() => {
      process.exit(0);
    });
  } catch (error) {
    process.exit(0);
  }
}

// Handle messages from main process
process.on("message", (message) => {
  if (!message || typeof message !== "object") return;
  
  if (message.type === "stop") {
    cleanup();
  }
});

process.on("SIGINT", cleanup);
process.on("SIGTERM", cleanup);
process.on("disconnect", cleanup);

// Main entry point
async function main() {
  log(`Starting scale worker - Port: ${PORT}, Baud: ${BAUD}, Poll: ${POLL_MS}ms`);
  
  sendUpdate({
    ready: null,
    weight: null,
    rawWeight: null,
    error: null,
    port: PORT,
  });
  
  const connected = await connect();
  
  if (connected) {
    startPolling();
  } else {
    scheduleReconnect();
  }
}

main().catch((error) => {
  console.error("[ScaleWorker] Unexpected error:", error);
  sendError(error.message || String(error));
  process.exitCode = 1;
});


