﻿const { app, BrowserWindow, ipcMain, screen } = require('electron');
const path = require('path');
const os = require('os');
const fs = require('fs');
const wifi = require('node-wifi');
const { fork } = require('child_process');

// Inicializa node-wifi usando la interfaz por defecto
wifi.init({ iface: null });

const isWindows = process.platform === 'win32';

function shouldDisableGpu() {
  if (process.argv.includes('--disable-gpu')) {
    return 'CLI flag --disable-gpu';
  }
  if (process.env.DISABLE_GPU === '1') {
    return 'DISABLE_GPU=1';
  }
  if (process.env.DISABLE_GPU === '0') {
    return null;
  }

  const isLinux = process.platform === 'linux';
  const isArm = process.arch.startsWith('arm');
  const runningHeadless = isLinux && !process.env.DISPLAY;

  if (isLinux && (isArm || runningHeadless)) {
    return 'Linux device sin GPU fiable (auto-detectado)';
  }

  return null;
}

const disableGpuReason = shouldDisableGpu();
if (disableGpuReason) {
  console.log(`[start] Deshabilitando aceleración GPU: ${disableGpuReason}`);
  app.disableHardwareAcceleration();
}

// Config file path for persistent settings
const CONFIG_FILE = path.join(app.getPath('userData'), 'scale-config.json');

function loadScaleConfig() {
  try {
    if (fs.existsSync(CONFIG_FILE)) {
      const data = fs.readFileSync(CONFIG_FILE, 'utf8');
      return JSON.parse(data);
    }
  } catch (error) {
    console.error('[Config] Error loading scale config:', error);
  }
  return {};
}

function saveScaleConfig(config) {
  try {
    fs.writeFileSync(CONFIG_FILE, JSON.stringify(config, null, 2), 'utf8');
    return true;
  } catch (error) {
    console.error('[Config] Error saving scale config:', error);
    return false;
  }
}

function resolvePositiveInt(rawValue, fallback) {
  const parsed = Number.parseInt(rawValue, 10);
  if (Number.isFinite(parsed) && parsed > 0) {
    return parsed;
  }
  return fallback;
}

const RFID_CONFIG = {
  path: process.env.RFID_PORT || (isWindows ? 'COM4' : '/dev/ttyUSB0'),
  baudRate: resolvePositiveInt(process.env.RFID_BAUD_RATE, 115200),
  interval: Math.max(resolvePositiveInt(process.env.RFID_INTERVAL, 500), 250),
  reconnectInterval: Math.max(
    resolvePositiveInt(process.env.RFID_RECONNECT_INTERVAL, 3000),
    1000
  ),
  debug: process.env.RFID_DEBUG === '1',
  execPath: process.env.RFID_NODE_PATH || 'node',
};

// Load saved scale config
const savedScaleConfig = loadScaleConfig();

const SCALE_CONFIG = {
  port: savedScaleConfig.port || process.env.SCALE_PORT || (isWindows ? 'COM6' : '/dev/ttyUSB1'),
  baudRate: resolvePositiveInt(savedScaleConfig.baudRate || process.env.SCALE_BAUD, 9600),
  slaveId: resolvePositiveInt(savedScaleConfig.slaveId || process.env.SCALE_SLAVE_ID, 1),
  startAddr: resolvePositiveInt(savedScaleConfig.startAddr || process.env.SCALE_START_ADDR, 0),
  pollInterval: Math.max(resolvePositiveInt(savedScaleConfig.pollInterval || process.env.SCALE_POLL_MS, 500), 100),
  wordOrder: (savedScaleConfig.wordOrder || process.env.SCALE_WORD_ORDER || 'BA').toUpperCase(),
  registers: (savedScaleConfig.registers || process.env.SCALE_REGISTERS || 'holding').toLowerCase(),
  divisor: parseFloat(savedScaleConfig.divisor || process.env.SCALE_DIV || '10000'),
  debug: process.env.SCALE_DEBUG === '1',
  execPath: process.env.SCALE_NODE_PATH || 'node',
};

/** @type {BrowserWindow | null} */
let mainWindow = null;
let rfidWorker = null;
let rfidReconnectTimer = null;
let scaleWorker = null;
let scaleReconnectTimer = null;
let isAppQuitting = false;

const rfidState = {
  ready: null,
  tag: null,
  updatedAt: null,
  error: null,
  port: RFID_CONFIG.path,
};

const scaleState = {
  ready: null,
  weight: null,
  rawWeight: null,
  updatedAt: null,
  error: null,
  port: SCALE_CONFIG.port,
};

function hasValidAddress(address) {
  if (!address || typeof address !== 'string') {
    return false;
  }
  if (address === '0.0.0.0' || address === '127.0.0.1') {
    return false;
  }
  if (address.startsWith('169.254.')) {
    return false;
  }
  return true;
}

function detectNetworkConnectivity() {
  const interfaces = os.networkInterfaces();
  const fallback = { hasNetwork: false };

  if (!interfaces || typeof interfaces !== 'object') {
    return fallback;
  }

  for (const [name, infos] of Object.entries(interfaces)) {
    if (!Array.isArray(infos)) {
      continue;
    }

    const validInfo = infos.find(
      (info) =>
        info &&
        info.family === 'IPv4' &&
        hasValidAddress(info.address) &&
        !info.internal
    );

    if (!validInfo) {
      continue;
    }

    const normalizedName = name.toLowerCase();
    const isWifiInterface =
      normalizedName.includes('wi-fi') ||
      normalizedName.includes('wifi') ||
      normalizedName.includes('wlan') ||
      normalizedName.includes('wireless');

    return {
      hasNetwork: true,
      interfaceName: name,
      type: isWifiInterface ? 'wifi' : 'ethernet',
    };
  }

  return fallback;
}

function notifyRfidUpdate() {
  if (!mainWindow || mainWindow.isDestroyed()) {
    return;
  }
  mainWindow.webContents.send('rfid:update', { ...rfidState });
}

function notifyScaleUpdate() {
  if (!mainWindow || mainWindow.isDestroyed()) {
    return;
  }
  mainWindow.webContents.send('scale:update', { ...scaleState });
}

function updateScaleState(patch) {
  if (!patch || typeof patch !== 'object') {
    return;
  }

  let changed = false;
  for (const [key, value] of Object.entries(patch)) {
    if (!(key in scaleState)) {
      continue;
    }
    if (!Object.is(scaleState[key], value)) {
      scaleState[key] = value;
      changed = true;
    }
  }

  if (changed) {
    notifyScaleUpdate();
  }
}

function scheduleScaleReconnect() {
  if (isAppQuitting) {
    return;
  }
  if (scaleReconnectTimer) {
    return;
  }
  scaleReconnectTimer = setTimeout(() => {
    scaleReconnectTimer = null;
    startScaleReader();
  }, 3000);
}

function clearScaleTimers() {
  if (scaleReconnectTimer) {
    clearTimeout(scaleReconnectTimer);
    scaleReconnectTimer = null;
  }
}

function handleScaleWorkerMessage(message) {
  if (!message || typeof message !== 'object') {
    return;
  }

  if (message.type === 'update' && message.payload) {
    updateScaleState(message.payload);
    return;
  }

  if (message.type === 'log' && SCALE_CONFIG.debug && message.message) {
    console.log('[Scale worker]', message.message);
    return;
  }

  if (message.type === 'error') {
    const errorText = message.error ? String(message.error) : 'Error en báscula';
    if (SCALE_CONFIG.debug) {
      console.error('[Scale worker] Error:', errorText);
    }
    updateScaleState({ ready: false, error: errorText });
  }
}

function startScaleReader() {
  if (isAppQuitting) {
    return;
  }
  if (scaleWorker) {
    return;
  }

  try {
    scaleWorker = fork(path.join(__dirname, 'scaleWorker.js'), [], {
      stdio: ['inherit', 'inherit', 'inherit', 'ipc'],
      execPath: SCALE_CONFIG.execPath,
      env: {
        ...process.env,
        SCALE_PORT: SCALE_CONFIG.port,
        SCALE_BAUD: String(SCALE_CONFIG.baudRate),
        SCALE_SLAVE_ID: String(SCALE_CONFIG.slaveId),
        SCALE_START_ADDR: String(SCALE_CONFIG.startAddr),
        SCALE_POLL_MS: String(SCALE_CONFIG.pollInterval),
        SCALE_WORD_ORDER: SCALE_CONFIG.wordOrder,
        SCALE_REGISTERS: SCALE_CONFIG.registers,
        SCALE_DIV: String(SCALE_CONFIG.divisor),
        SCALE_DEBUG: SCALE_CONFIG.debug ? '1' : '0',
      },
    });
  } catch (error) {
    console.error('[Scale] No se pudo iniciar el proceso lector:', error);
    updateScaleState({ ready: false, error: error.message });
    scheduleScaleReconnect();
    return;
  }

  updateScaleState({ ready: null, error: null, weight: null, port: SCALE_CONFIG.port });

  scaleWorker.on('message', handleScaleWorkerMessage);
  scaleWorker.on('error', (error) => {
    console.error('[Scale] Error en el proceso lector:', error);
  });
  scaleWorker.on('exit', (code, signal) => {
    if (SCALE_CONFIG.debug) {
      console.log('[Scale] Proceso lector finalizado', { code, signal });
    }
    scaleWorker = null;
    if (!isAppQuitting) {
      updateScaleState({ ready: false });
      scheduleScaleReconnect();
    }
  });
}

function stopScaleReader() {
  clearScaleTimers();
  const worker = scaleWorker;
  scaleWorker = null;
  if (worker) {
    try {
      worker.removeAllListeners();
      worker.kill();
    } catch (error) {
      console.error('[Scale] No se pudo finalizar el proceso lector:', error);
    }
  }
}

function restartScaleReader() {
  stopScaleReader();
  // Small delay before restarting
  setTimeout(() => {
    startScaleReader();
  }, 500);
}

function updateRfidState(patch) {
  if (!patch || typeof patch !== 'object') {
    return;
  }

  let changed = false;
  for (const [key, value] of Object.entries(patch)) {
    if (!(key in rfidState)) {
      continue;
    }
    if (!Object.is(rfidState[key], value)) {
      rfidState[key] = value;
      changed = true;
    }
  }

  if (changed) {
    notifyRfidUpdate();
  }
}

function scheduleRfidReconnect() {
  if (isAppQuitting) {
    return;
  }
  if (rfidReconnectTimer) {
    return;
  }
  rfidReconnectTimer = setTimeout(() => {
    rfidReconnectTimer = null;
    startRfidReader();
  }, RFID_CONFIG.reconnectInterval);
}

function clearRfidTimers() {
  if (rfidReconnectTimer) {
    clearTimeout(rfidReconnectTimer);
    rfidReconnectTimer = null;
  }
}

function handleRfidWorkerMessage(message) {
  if (!message || typeof message !== 'object') {
    return;
  }

  if (message.type === 'update' && message.payload) {
    updateRfidState(message.payload);
    return;
  }

  if (message.type === 'log' && RFID_CONFIG.debug && message.message) {
    console.log('[RFID worker]', message.message);
    return;
  }

  if (message.type === 'error') {
    const errorText = message.error ? String(message.error) : 'Error en lector RFID';
    if (RFID_CONFIG.debug) {
      console.error('[RFID worker] Error:', errorText);
    }
    updateRfidState({ ready: false, error: errorText });
  }
}

function startRfidReader() {
  if (isAppQuitting) {
    return;
  }
  if (rfidWorker) {
    return;
  }

  try {
    rfidWorker = fork(path.join(__dirname, 'rfidWorker.js'), [], {
      stdio: ['inherit', 'inherit', 'inherit', 'ipc'],
      execPath: RFID_CONFIG.execPath,
      env: {
        ...process.env,
        RFID_PORT: RFID_CONFIG.path,
        RFID_BAUD_RATE: String(RFID_CONFIG.baudRate),
        RFID_INTERVAL: String(RFID_CONFIG.interval),
        RFID_RECONNECT_INTERVAL: String(RFID_CONFIG.reconnectInterval),
        RFID_DEBUG: RFID_CONFIG.debug ? '1' : '0',
      },
    });
  } catch (error) {
    console.error('[RFID] No se pudo iniciar el proceso lector:', error);
    updateRfidState({ ready: false, error: error.message });
    scheduleRfidReconnect();
    return;
  }

  updateRfidState({ ready: null, error: null, tag: null, port: RFID_CONFIG.path });

  rfidWorker.on('message', handleRfidWorkerMessage);
  rfidWorker.on('error', (error) => {
    console.error('[RFID] Error en el proceso lector:', error);
  });
  rfidWorker.on('exit', (code, signal) => {
    if (RFID_CONFIG.debug) {
      console.log('[RFID] Proceso lector finalizado', { code, signal });
    }
    rfidWorker = null;
    if (!isAppQuitting) {
      updateRfidState({ ready: false });
      scheduleRfidReconnect();
    }
  });
}

function stopRfidReader() {
  clearRfidTimers();
  const worker = rfidWorker;
  rfidWorker = null;
  if (worker) {
    try {
      worker.removeAllListeners();
      worker.kill();
    } catch (error) {
      console.error('[RFID] No se pudo finalizar el proceso lector:', error);
    }
  }
}

function createMainWindow() {
  // Configuración fija para pantalla 640x480
  const windowWidth = 640;
  const windowHeight = 480;

  mainWindow = new BrowserWindow({
    width: windowWidth,
    height: windowHeight,
    resizable: false,
    fullscreen: true,
    kiosk: false,
    autoHideMenuBar: true,
    backgroundColor: '#101419',
    webPreferences: {
      preload: path.join(__dirname, 'preload.js'),
      nodeIntegration: false,
      contextIsolation: true,
      sandbox: false
    }
  });

  mainWindow.loadFile(path.join(__dirname, '../public/index.html'));

  mainWindow.on('closed', () => {
    mainWindow = null;
  });

  mainWindow.webContents.on('did-finish-load', () => {
    notifyRfidUpdate();
notifyScaleUpdate();
  });
}

app.whenReady().then(() => {
  startRfidReader();
  startScaleReader();
  createMainWindow();
});

app.on('activate', () => {
  if (BrowserWindow.getAllWindows().length === 0) {
    createMainWindow();
  }
});

app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') {
    app.quit();
  }
});

app.on('before-quit', () => {
  isAppQuitting = true;
  stopRfidReader();
  stopScaleReader();
});

ipcMain.handle('wifi:get-status', async () => {
  try {
    const connections = await wifi.getCurrentConnections();
    const first = connections?.find((conn) => conn?.ssid);
    const networkStatus = detectNetworkConnectivity();

    if (first && first.ssid) {
      return {
        connected: true,
        ssid: first.ssid,
        hasNetwork: true,
        type: 'wifi',
        interfaceName: networkStatus.interfaceName ?? null,
      };
    }

    return {
      connected: false,
      hasNetwork: Boolean(networkStatus.hasNetwork),
      type: networkStatus.type ?? null,
      interfaceName: networkStatus.interfaceName ?? null,
    };
  } catch (error) {
    const networkStatus = detectNetworkConnectivity();
    return {
      connected: false,
      hasNetwork: Boolean(networkStatus.hasNetwork),
      type: networkStatus.type ?? null,
      interfaceName: networkStatus.interfaceName ?? null,
      error: error.message || String(error),
    };
  }
});

ipcMain.handle('rfid:get-status', async () => {
  return { ...rfidState };
});

ipcMain.handle('wifi:scan', async () => {
  try {
    const networks = await wifi.scan();
    const sorted = networks
      .filter((net) => net.ssid)
      .sort((a, b) => (b.signal_level || 0) - (a.signal_level || 0));
    return { ok: true, networks: sorted };
  } catch (error) {
    return { ok: false, error: error.message || String(error) };
  }
});

ipcMain.handle('wifi:connect', async (_event, payload) => {
  const { ssid, password } = payload || {};
  if (!ssid || typeof ssid !== 'string') {
    return { ok: false, error: 'SSID invalido' };
  }
  try {
    await wifi.connect({ ssid, password: password ?? undefined });
    return { ok: true };
  } catch (error) {
    return { ok: false, error: error.message || String(error) };
  }
});

// Scale IPC handlers
ipcMain.handle('scale:get-status', async () => {
  return { ...scaleState };
});

ipcMain.handle('scale:get-config', async () => {
  return {
    port: SCALE_CONFIG.port,
    baudRate: SCALE_CONFIG.baudRate,
    slaveId: SCALE_CONFIG.slaveId,
    startAddr: SCALE_CONFIG.startAddr,
    pollInterval: SCALE_CONFIG.pollInterval,
    wordOrder: SCALE_CONFIG.wordOrder,
    registers: SCALE_CONFIG.registers,
    divisor: SCALE_CONFIG.divisor,
  };
});

ipcMain.handle('scale:set-config', async (_event, payload) => {
  if (!payload || typeof payload !== 'object') {
    return { ok: false, error: 'Configuración inválida' };
  }

  // Update config
  if (payload.port && typeof payload.port === 'string') {
    SCALE_CONFIG.port = payload.port;
  }
  if (payload.baudRate && Number.isFinite(payload.baudRate)) {
    SCALE_CONFIG.baudRate = payload.baudRate;
  }
  if (payload.slaveId && Number.isFinite(payload.slaveId)) {
    SCALE_CONFIG.slaveId = payload.slaveId;
  }
  if (payload.startAddr !== undefined && Number.isFinite(payload.startAddr)) {
    SCALE_CONFIG.startAddr = payload.startAddr;
  }
  if (payload.pollInterval && Number.isFinite(payload.pollInterval)) {
    SCALE_CONFIG.pollInterval = Math.max(payload.pollInterval, 100);
  }
  if (payload.wordOrder && typeof payload.wordOrder === 'string') {
    SCALE_CONFIG.wordOrder = payload.wordOrder.toUpperCase();
  }
  if (payload.registers && typeof payload.registers === 'string') {
    SCALE_CONFIG.registers = payload.registers.toLowerCase();
  }
  if (payload.divisor && Number.isFinite(payload.divisor)) {
    SCALE_CONFIG.divisor = payload.divisor;
  }

  // Save to file
  const saved = saveScaleConfig({
    port: SCALE_CONFIG.port,
    baudRate: SCALE_CONFIG.baudRate,
    slaveId: SCALE_CONFIG.slaveId,
    startAddr: SCALE_CONFIG.startAddr,
    pollInterval: SCALE_CONFIG.pollInterval,
    wordOrder: SCALE_CONFIG.wordOrder,
    registers: SCALE_CONFIG.registers,
    divisor: SCALE_CONFIG.divisor,
  });

  // Restart scale reader with new config
  restartScaleReader();

  return { ok: true, saved };
});

ipcMain.handle('scale:list-ports', async () => {
  try {
    const { SerialPort } = require('serialport');
    const ports = await SerialPort.list();
    return {
      ok: true,
      ports: ports.map((p) => ({
        path: p.path,
        manufacturer: p.manufacturer || null,
        serialNumber: p.serialNumber || null,
        pnpId: p.pnpId || null,
        friendlyName: p.friendlyName || null,
      })),
    };
  } catch (error) {
    return { ok: false, error: error.message || String(error), ports: [] };
  }
});
