﻿const { app, BrowserWindow, ipcMain, screen } = require('electron');
const path = require('path');
const os = require('os');
const wifi = require('node-wifi');
const { fork } = require('child_process');

// Inicializa node-wifi usando la interfaz por defecto
wifi.init({ iface: null });

const isWindows = process.platform === 'win32';

function resolvePositiveInt(rawValue, fallback) {
  const parsed = Number.parseInt(rawValue, 10);
  if (Number.isFinite(parsed) && parsed > 0) {
    return parsed;
  }
  return fallback;
}

const RFID_CONFIG = {
  path: process.env.RFID_PORT || (isWindows ? 'COM4' : '/dev/ttyUSB0'),
  baudRate: resolvePositiveInt(process.env.RFID_BAUD_RATE, 115200),
  interval: Math.max(resolvePositiveInt(process.env.RFID_INTERVAL, 500), 250),
  reconnectInterval: Math.max(
    resolvePositiveInt(process.env.RFID_RECONNECT_INTERVAL, 3000),
    1000
  ),
  debug: process.env.RFID_DEBUG === '1',
  execPath: process.env.RFID_NODE_PATH || 'node',
};

/** @type {BrowserWindow | null} */
let mainWindow = null;
let rfidWorker = null;
let rfidReconnectTimer = null;
let isAppQuitting = false;

const rfidState = {
  ready: null,
  tag: null,
  updatedAt: null,
  error: null,
  port: RFID_CONFIG.path,
};

function hasValidAddress(address) {
  if (!address || typeof address !== 'string') {
    return false;
  }
  if (address === '0.0.0.0' || address === '127.0.0.1') {
    return false;
  }
  if (address.startsWith('169.254.')) {
    return false;
  }
  return true;
}

function detectNetworkConnectivity() {
  const interfaces = os.networkInterfaces();
  const fallback = { hasNetwork: false };

  if (!interfaces || typeof interfaces !== 'object') {
    return fallback;
  }

  for (const [name, infos] of Object.entries(interfaces)) {
    if (!Array.isArray(infos)) {
      continue;
    }

    const validInfo = infos.find(
      (info) =>
        info &&
        info.family === 'IPv4' &&
        hasValidAddress(info.address) &&
        !info.internal
    );

    if (!validInfo) {
      continue;
    }

    const normalizedName = name.toLowerCase();
    const isWifiInterface =
      normalizedName.includes('wi-fi') ||
      normalizedName.includes('wifi') ||
      normalizedName.includes('wlan') ||
      normalizedName.includes('wireless');

    return {
      hasNetwork: true,
      interfaceName: name,
      type: isWifiInterface ? 'wifi' : 'ethernet',
    };
  }

  return fallback;
}

function notifyRfidUpdate() {
  if (!mainWindow || mainWindow.isDestroyed()) {
    return;
  }
  mainWindow.webContents.send('rfid:update', { ...rfidState });
}

function updateRfidState(patch) {
  if (!patch || typeof patch !== 'object') {
    return;
  }

  let changed = false;
  for (const [key, value] of Object.entries(patch)) {
    if (!(key in rfidState)) {
      continue;
    }
    if (!Object.is(rfidState[key], value)) {
      rfidState[key] = value;
      changed = true;
    }
  }

  if (changed) {
    notifyRfidUpdate();
  }
}

function scheduleRfidReconnect() {
  if (isAppQuitting) {
    return;
  }
  if (rfidReconnectTimer) {
    return;
  }
  rfidReconnectTimer = setTimeout(() => {
    rfidReconnectTimer = null;
    startRfidReader();
  }, RFID_CONFIG.reconnectInterval);
}

function clearRfidTimers() {
  if (rfidReconnectTimer) {
    clearTimeout(rfidReconnectTimer);
    rfidReconnectTimer = null;
  }
}

function handleRfidWorkerMessage(message) {
  if (!message || typeof message !== 'object') {
    return;
  }

  if (message.type === 'update' && message.payload) {
    updateRfidState(message.payload);
    return;
  }

  if (message.type === 'log' && RFID_CONFIG.debug && message.message) {
    console.log('[RFID worker]', message.message);
    return;
  }

  if (message.type === 'error') {
    const errorText = message.error ? String(message.error) : 'Error en lector RFID';
    if (RFID_CONFIG.debug) {
      console.error('[RFID worker] Error:', errorText);
    }
    updateRfidState({ ready: false, error: errorText });
  }
}

function startRfidReader() {
  if (isAppQuitting) {
    return;
  }
  if (rfidWorker) {
    return;
  }

  try {
    rfidWorker = fork(path.join(__dirname, 'rfidWorker.js'), [], {
      stdio: ['inherit', 'inherit', 'inherit', 'ipc'],
      execPath: RFID_CONFIG.execPath,
      env: {
        ...process.env,
        RFID_PORT: RFID_CONFIG.path,
        RFID_BAUD_RATE: String(RFID_CONFIG.baudRate),
        RFID_INTERVAL: String(RFID_CONFIG.interval),
        RFID_RECONNECT_INTERVAL: String(RFID_CONFIG.reconnectInterval),
        RFID_DEBUG: RFID_CONFIG.debug ? '1' : '0',
      },
    });
  } catch (error) {
    console.error('[RFID] No se pudo iniciar el proceso lector:', error);
    updateRfidState({ ready: false, error: error.message });
    scheduleRfidReconnect();
    return;
  }

  updateRfidState({ ready: null, error: null, tag: null, port: RFID_CONFIG.path });

  rfidWorker.on('message', handleRfidWorkerMessage);
  rfidWorker.on('error', (error) => {
    console.error('[RFID] Error en el proceso lector:', error);
  });
  rfidWorker.on('exit', (code, signal) => {
    if (RFID_CONFIG.debug) {
      console.log('[RFID] Proceso lector finalizado', { code, signal });
    }
    rfidWorker = null;
    if (!isAppQuitting) {
      updateRfidState({ ready: false });
      scheduleRfidReconnect();
    }
  });
}

function stopRfidReader() {
  clearRfidTimers();
  const worker = rfidWorker;
  rfidWorker = null;
  if (worker) {
    try {
      worker.removeAllListeners();
      worker.kill();
    } catch (error) {
      console.error('[RFID] No se pudo finalizar el proceso lector:', error);
    }
  }
}

function createMainWindow() {
  // Obtener las dimensiones de la pantalla principal
  const primaryDisplay = screen.getPrimaryDisplay();
  const { width, height } = primaryDisplay.workAreaSize;

  mainWindow = new BrowserWindow({
    width: width,
    height: height,
    resizable: true,
    fullscreen: true,
    kiosk: false,
    autoHideMenuBar: true,
    backgroundColor: '#101419',
    webPreferences: {
      preload: path.join(__dirname, 'preload.js'),
      nodeIntegration: false,
      contextIsolation: true,
      sandbox: false
    }
  });

  mainWindow.loadFile(path.join(__dirname, '../public/index.html'));

  mainWindow.on('closed', () => {
    mainWindow = null;
  });

  mainWindow.webContents.on('did-finish-load', () => {
    notifyRfidUpdate();
  });
}

app.whenReady().then(() => {
  startRfidReader();
  createMainWindow();
});

app.on('activate', () => {
  if (BrowserWindow.getAllWindows().length === 0) {
    createMainWindow();
  }
});

app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') {
    app.quit();
  }
});

app.on('before-quit', () => {
  isAppQuitting = true;
  stopRfidReader();
});

ipcMain.handle('wifi:get-status', async () => {
  try {
    const connections = await wifi.getCurrentConnections();
    const first = connections?.find((conn) => conn?.ssid);
    const networkStatus = detectNetworkConnectivity();

    if (first && first.ssid) {
      return {
        connected: true,
        ssid: first.ssid,
        hasNetwork: true,
        type: 'wifi',
        interfaceName: networkStatus.interfaceName ?? null,
      };
    }

    return {
      connected: false,
      hasNetwork: Boolean(networkStatus.hasNetwork),
      type: networkStatus.type ?? null,
      interfaceName: networkStatus.interfaceName ?? null,
    };
  } catch (error) {
    const networkStatus = detectNetworkConnectivity();
    return {
      connected: false,
      hasNetwork: Boolean(networkStatus.hasNetwork),
      type: networkStatus.type ?? null,
      interfaceName: networkStatus.interfaceName ?? null,
      error: error.message || String(error),
    };
  }
});

ipcMain.handle('rfid:get-status', async () => {
  return { ...rfidState };
});

ipcMain.handle('wifi:scan', async () => {
  try {
    const networks = await wifi.scan();
    const sorted = networks
      .filter((net) => net.ssid)
      .sort((a, b) => (b.signal_level || 0) - (a.signal_level || 0));
    return { ok: true, networks: sorted };
  } catch (error) {
    return { ok: false, error: error.message || String(error) };
  }
});

ipcMain.handle('wifi:connect', async (_event, payload) => {
  const { ssid, password } = payload || {};
  if (!ssid || typeof ssid !== 'string') {
    return { ok: false, error: 'SSID invalido' };
  }
  try {
    await wifi.connect({ ssid, password: password ?? undefined });
    return { ok: true };
  } catch (error) {
    return { ok: false, error: error.message || String(error) };
  }
});
