"use strict";

const SIMULATED_WEIGHT = 42.1;
const API_ENDPOINT = "https://tipgreen-api.deploy.ecoquimic.com/containermeasurement/rfid";
const RFID_WAITING_LABEL = "Esperando RFID";
const RFID_ERROR_LABEL = "RFID no disponible";

const screens = {
	measure: document.getElementById("screen-measure"),
	networks: document.getElementById("screen-networks"),
	password: document.getElementById("screen-password"),
};

const versionEl = document.getElementById("app-version");
const connectionLabel = document.getElementById("connection-label");
const openWifiBtn = document.getElementById("open-wifi");

const rfidDisplay = document.getElementById("rfid-display");
const weightDisplay = document.getElementById("weight-display");
const sendBtn = document.getElementById("send-btn");
const measureFeedbackEl = document.getElementById("measure-feedback");
const measureDisplay = document.getElementById("measure-display");
const measureDetails = document.getElementById("measure-details");
const resultDateEl = document.getElementById("result-date");
const resultContainerEl = document.getElementById("result-container");
const resultZoneEl = document.getElementById("result-zone");
const resultWeightEl = document.getElementById("result-weight");

let rfidStatus = {
	tag: null,
	updatedAt: null,
	ready: null,
	error: null,
	port: null,
};

let rfidUnsubscribe = null;

const networksMessageEl = document.getElementById("networks-message");
const networksListEl = document.getElementById("networks-list");
const networksRefreshBtn = document.getElementById("networks-refresh");
const networksBackBtn = document.getElementById("networks-back");
const networksFeedbackEl = document.getElementById("networks-feedback");

const passwordTitleEl = document.getElementById("password-title");
const passwordField = document.getElementById("password-field");
const passwordConnectBtn = document.getElementById("password-connect");
const passwordBackBtn = document.getElementById("password-back");
const passwordFeedbackEl = document.getElementById("password-feedback");
const keyboardContainer = document.getElementById("keyboard");

const KEY_ROWS = [
	["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"],
	["q", "w", "e", "r", "t", "y", "u", "i", "o", "p"],
	["a", "s", "d", "f", "g", "h", "j", "k", "l", "\u00F1"],
	["shift", "z", "x", "c", "v", "b", "n", "m", "@", "."],
	["space", "backspace", "clear"],
];

const ACTION_KEY_CONFIG = {
	shift: { label: "Mayus", className: "key-btn--wide" },
	space: { label: "Espacio", className: "key-btn--space" },
	backspace: { label: "Borrar", className: "key-btn--wide" },
	clear: { label: "Reset", className: "key-btn--wide" },
};

const NETWORK_MANAGER_HINT = [
	"NetworkManager no esta activo.",
	"Ejecuta:",
	"sudo systemctl enable --now NetworkManager",
	"sudo systemctl restart NetworkManager",
].join("\n");

let availableNetworks = [];
let selectedNetwork = null;
let shiftEnabled = false;
let keyboardBuilt = false;

function setScreen(name) {
	Object.entries(screens).forEach(([key, element]) => {
		if (!element) {
			return;
		}
		if (key === name) {
			element.classList.remove("hidden");
		} else {
			element.classList.add("hidden");
		}
	});
}

function setFeedback(element, message = "", isError = false) {
	if (!element) {
		return;
	}
	element.textContent = message || "";
	element.classList.toggle("feedback--error", Boolean(isError));
}

function renderRfidState() {
	if (!rfidDisplay) {
		return;
	}

	if (rfidStatus.tag) {
		rfidDisplay.textContent = rfidStatus.tag;
		return;
	}

	if (rfidStatus.ready === false) {
		rfidDisplay.textContent = RFID_ERROR_LABEL;
		return;
	}

	rfidDisplay.textContent = RFID_WAITING_LABEL;
}

function applyRfidStatus(status) {
	if (!status || typeof status !== "object") {
		return;
	}

	if ("tag" in status) {
		const value = typeof status.tag === "string" && status.tag.trim().length ? status.tag.trim().toUpperCase() : null;
		rfidStatus.tag = value;
	}

	if ("updatedAt" in status) {
		const value = Number.isFinite(status.updatedAt) ? status.updatedAt : null;
		rfidStatus.updatedAt = value;
	}

	if ("ready" in status) {
		rfidStatus.ready = status.ready === null ? null : Boolean(status.ready);
	}

	if ("error" in status) {
		rfidStatus.error = status.error ? String(status.error) : null;
	}

	if ("port" in status) {
		rfidStatus.port = status.port ? String(status.port) : null;
	}

	renderRfidState();
}

function updateConnectionLabel(status) {
	if (!connectionLabel) {
		return;
	}

	if (status?.connected && status?.ssid) {
		const ssid = status.ssid || "red desconocida";
		connectionLabel.textContent = `Conectado a ${ssid}`;
		connectionLabel.classList.remove("top-bar__connection--lost", "hidden");
		return;
	}

	if (status?.hasNetwork) {
		const byCable = status?.type === "ethernet";
		const interfaceLabel = status?.interfaceName ? ` (${status.interfaceName})` : "";
		connectionLabel.textContent = byCable ? `Conectado por cable${interfaceLabel}` : `Conectado${interfaceLabel}`;
		connectionLabel.classList.remove("top-bar__connection--lost", "hidden");
		return;
	}

	connectionLabel.textContent = "Sin conexion";
	connectionLabel.classList.add("top-bar__connection--lost");
	connectionLabel.classList.remove("hidden");
}

function renderWeightDisplay(weight) {
	if (!weightDisplay) {
		return;
	}

	const weightStr = String(weight);
	const hasDecimals = weightStr.includes(".");

	if (hasDecimals) {
		const [integerPart, decimalPart] = weightStr.split(".");
		weightDisplay.innerHTML = `<span class="measure-weight-integer">${integerPart}</span><span class="measure-weight-decimal">.${decimalPart}</span>`;
		weightDisplay.classList.add("measure-weight--with-decimals");
	} else {
		weightDisplay.textContent = weightStr;
		weightDisplay.classList.remove("measure-weight--with-decimals");
	}
}

function renderMeasureView() {
	renderRfidState();
	renderWeightDisplay(SIMULATED_WEIGHT);
	setFeedback(measureFeedbackEl, "", false);
	measureDisplay?.classList.remove("measure-display--compact");
	measureDetails?.classList.add("hidden");
	setScreen("measure");
}

function needsPassword(network) {
	const raw = (network?.security || "").toString().toLowerCase();
	if (!raw || raw.includes("open") || raw.includes("none") || raw.includes("sin")) {
		return false;
	}
	return true;
}

function getSignalLabel(network) {
	const quality = Number.isFinite(network?.quality) ? Number(network.quality) : Number(network?.signal_level);
	if (!Number.isFinite(quality)) {
		return "";
	}
	const normalized = quality > 0 && quality <= 100 ? quality : Math.min(Math.max(quality + 100, 0), 100);
	return `${Math.round(normalized)}%`;
}

function renderNetworksLoading(message = "Buscando redes...") {
	networksMessageEl.textContent = message;
	networksListEl.innerHTML = "";
	setFeedback(networksFeedbackEl, "", false);
	setScreen("networks");
}

function renderNetworks(networks) {
	availableNetworks = Array.isArray(networks) ? networks : [];
	networksListEl.innerHTML = "";

	if (!availableNetworks.length) {
		networksMessageEl.textContent = "Sin redes disponibles";
		return;
	}

	networksMessageEl.textContent = "Toca una red";

	availableNetworks.forEach((network, index) => {
		if (!network?.ssid) {
			return;
		}
		const button = document.createElement("button");
		button.type = "button";
		button.className = "network-btn";

		const ssidLabel = document.createElement("strong");
		ssidLabel.textContent = network.ssid;

		const signal = document.createElement("span");
		signal.textContent = getSignalLabel(network);

		button.append(ssidLabel, signal);
		button.addEventListener("click", () => handleNetworkChoice(index, button));
		networksListEl.append(button);
	});
}

function handleNetworkChoice(index, button) {
	const network = availableNetworks[index];
	if (!network?.ssid) {
		return;
	}
	selectedNetwork = network;

	if (needsPassword(network)) {
		showPasswordScreen(network);
		return;
	}

	connectToNetwork(
		{ ssid: network.ssid },
		{
			feedbackEl: networksFeedbackEl,
			buttonToDisable: button,
		}
	);
}

function showPasswordScreen(network) {
	passwordTitleEl.textContent = network?.ssid || "Introduce la contrasena";
	passwordField.value = "";
	setFeedback(passwordFeedbackEl, "", false);
	shiftEnabled = false;
	updateShiftVisual();
	buildKeyboard();
	setScreen("password");

	setTimeout(() => {
		passwordField.focus();
	}, 120);
}

function buildKeyboard() {
	if (keyboardBuilt || !keyboardContainer) {
		return;
	}

	KEY_ROWS.forEach((rowKeys) => {
		const row = document.createElement("div");
		row.className = "keyboard__row";

		rowKeys.forEach((key) => {
			const button = document.createElement("button");
			button.type = "button";
			button.classList.add("key-btn");

			const config = ACTION_KEY_CONFIG[key];
			if (config) {
				button.dataset.action = key;
				button.textContent = config.label;
				if (config.className) {
					button.classList.add(config.className);
				}
			} else {
				button.dataset.value = key;
				const label = /^[a-z\u00F1]$/i.test(key) ? key.toUpperCase() : key;
				button.textContent = label;
			}

			button.addEventListener("click", () => {
				handleKeyPress(button.dataset.action, button.dataset.value);
			});

			row.append(button);
		});

		keyboardContainer.append(row);
	});

	keyboardBuilt = true;
}

async function initializeRfid() {
	if (!window.rfidApi) {
		console.warn("rfidApi no esta disponible en el contexto de la ventana.");
		applyRfidStatus({ ready: false, error: "RFID no disponible" });
		return;
	}

	try {
		const status = await window.rfidApi.getStatus();
		applyRfidStatus(status);
	} catch (error) {
		console.warn("No se pudo obtener el estado RFID", error);
		applyRfidStatus({
			ready: false,
			error: error?.message || String(error),
		});
	}

	if (typeof window.rfidApi.onUpdate === "function") {
		if (typeof rfidUnsubscribe === "function") {
			rfidUnsubscribe();
		}
		rfidUnsubscribe = window.rfidApi.onUpdate((status) => {
			applyRfidStatus(status);
		});
	}
}

function updateShiftVisual() {
	if (!keyboardContainer) {
		return;
	}
	const shiftKey = keyboardContainer.querySelector('[data-action="shift"]');
	if (shiftKey) {
		shiftKey.classList.toggle("key-btn--active", shiftEnabled);
	}
}

function handleKeyPress(action, value) {
	if (!passwordField) {
		return;
	}

	if (action === "shift") {
		shiftEnabled = !shiftEnabled;
		updateShiftVisual();
		return;
	}

	if (action === "space") {
		passwordField.value += " ";
		passwordField.focus();
		return;
	}

	if (action === "backspace") {
		passwordField.value = passwordField.value.slice(0, -1);
		passwordField.focus();
		return;
	}

	if (action === "clear") {
		passwordField.value = "";
		passwordField.focus();
		return;
	}

	if (!value) {
		return;
	}

	const character = shiftEnabled ? value.toUpperCase() : value;
	passwordField.value += character;
	passwordField.focus();

	if (shiftEnabled) {
		shiftEnabled = false;
		updateShiftVisual();
	}
}

async function loadNetworks() {
	networksRefreshBtn.disabled = true;
	setFeedback(networksFeedbackEl, "", false);

	try {
		const response = await window.wifiApi.scan();

		if (!response?.ok) {
			renderNetworks(response?.networks || []);
			if (response?.error) {
				setFeedback(networksFeedbackEl, formatError(response.error), true);
			}
			return;
		}

		renderNetworks(response.networks || []);
	} catch (error) {
		renderNetworks([]);
		setFeedback(networksFeedbackEl, formatError(error?.message || String(error)), true);
	} finally {
		networksRefreshBtn.disabled = false;
	}
}

async function connectToNetwork({ ssid, password }, options = {}) {
	if (!ssid) {
		return;
	}

	const { feedbackEl, buttonToDisable, onSuccess } = options;

	if (buttonToDisable) {
		buttonToDisable.disabled = true;
	}

	if (feedbackEl) {
		setFeedback(feedbackEl, "Conectando...", false);
	}

	try {
		const result = await window.wifiApi.connect({ ssid, password });

		if (result?.ok) {
			if (typeof onSuccess === "function") {
				onSuccess();
			}
			if (feedbackEl) {
				setFeedback(feedbackEl, "Conectado", false);
			}
			await new Promise((resolve) => setTimeout(resolve, 800));
			await refreshStatus({ skipNetworkScan: true });
			return;
		}

		const errorMessage = result?.error ? formatError(result.error) : "No se pudo conectar.";
		if (feedbackEl) {
			setFeedback(feedbackEl, errorMessage, true);
		}
	} catch (error) {
		if (feedbackEl) {
			setFeedback(feedbackEl, formatError(error?.message || String(error)), true);
		}
	} finally {
		if (buttonToDisable) {
			buttonToDisable.disabled = false;
		}
	}
}

function isNetworkManagerDown(message) {
	if (!message) {
		return false;
	}
	const normalized = message.toString().toLowerCase();
	return normalized.includes("networkmanager is not running") || normalized.includes("org.freedesktop.networkmanager");
}

function formatError(message) {
	if (!message) {
		return "Ha ocurrido un error desconocido.";
	}

	if (isNetworkManagerDown(message)) {
		return NETWORK_MANAGER_HINT;
	}

	return message;
}

async function refreshStatus(options = {}) {
	const { skipNetworkScan = false } = options;
	let status = null;
	try {
		status = await window.wifiApi.getStatus();
	} catch (error) {
		console.warn("No se pudo obtener el estado Wi-Fi", error);
	}

	updateConnectionLabel(status);

	if ((status?.connected && status?.ssid) || status?.hasNetwork) {
		renderMeasureView();
		return true;
	}

	if (!skipNetworkScan) {
		renderNetworksLoading("Buscando redes...");
		await loadNetworks();
	} else {
		renderNetworksLoading("Selecciona red");
	}
	return false;
}

async function sendMeasurement() {
	if (!rfidStatus.tag) {
		setFeedback(measureFeedbackEl, "Acerca un contenedor para leer el RFID.", true);
		return;
	}

	setFeedback(measureFeedbackEl, "Enviando datos...", false);
	sendBtn.disabled = true;

	try {
		const response = await fetch(API_ENDPOINT, {
			method: "POST",
			headers: {
				"Content-Type": "application/json",
			},
			body: JSON.stringify({
				rfid: rfidStatus.tag,
				totalWeight: SIMULATED_WEIGHT,
			}),
		});

		const payload = await safeJson(response);

		if (!response.ok) {
			const message = payload?.error || payload?.message || `Error ${response.status}: ${response.statusText}`;
			throw new Error(message);
		}

		updateMeasurementResult(payload);

		const statusMessage = response.status === 201 ? "Medicion enviada (201)." : `Medicion enviada (codigo ${response.status}).`;
		setFeedback(measureFeedbackEl, statusMessage, false);
	} catch (error) {
		setFeedback(measureFeedbackEl, formatError(error?.message || String(error)), true);
		try {
			await refreshStatus();
		} catch (_) {
			// ignore secondary errors
		}
	} finally {
		sendBtn.disabled = false;
	}
}

function attachEvents() {
	openWifiBtn?.addEventListener("click", () => {
		selectedNetwork = null;
		renderNetworksLoading("Buscando redes...");
		loadNetworks();
	});

	networksRefreshBtn?.addEventListener("click", () => {
		renderNetworksLoading("Actualizando redes...");
		loadNetworks();
	});

	networksBackBtn?.addEventListener("click", () => {
		setFeedback(networksFeedbackEl, "", false);
		renderMeasureView();
	});

	passwordConnectBtn?.addEventListener("click", () => {
		if (!selectedNetwork?.ssid) {
			return;
		}
		connectToNetwork(
			{ ssid: selectedNetwork.ssid, password: passwordField.value },
			{
				feedbackEl: passwordFeedbackEl,
				buttonToDisable: passwordConnectBtn,
				onSuccess: () => {
					passwordField.value = "";
				},
			}
		);
	});

	passwordBackBtn?.addEventListener("click", () => {
		selectedNetwork = null;
		passwordField.value = "";
		setFeedback(passwordFeedbackEl, "", false);
		shiftEnabled = false;
		updateShiftVisual();
		renderNetworksLoading("Selecciona red");
		renderNetworks(availableNetworks);
	});

	passwordField?.addEventListener("input", () => {
		setFeedback(passwordFeedbackEl, "", false);
	});

	passwordField?.addEventListener("keydown", (event) => {
		if (event.key === "Enter") {
			event.preventDefault();
			passwordConnectBtn?.click();
		}
		if (event.key === "Escape") {
			event.preventDefault();
			passwordBackBtn?.click();
		}
	});

	sendBtn?.addEventListener("click", () => {
		sendMeasurement();
	});
}

function showVersion() {
	try {
		const version = window.appApi?.getVersion?.();
		if (versionEl && version) {
			versionEl.textContent = `v${version}`;
		}
	} catch (error) {
		console.warn("No se pudo obtener la version de la app", error);
	}
}

function assertApiAvailable() {
	if (!window.wifiApi) {
		throw new Error("wifiApi no esta disponible en el contexto de la ventana.");
	}
	if (!window.appApi) {
		throw new Error("appApi no esta disponible en el contexto de la ventana.");
	}
}

async function init() {
	try {
		assertApiAvailable();
		showVersion();
		buildKeyboard();
		attachEvents();
		renderRfidState();
		await initializeRfid();
		await refreshStatus();
	} catch (error) {
		console.error(error);
		setFeedback(networksFeedbackEl, formatError(error?.message || String(error)), true);
		setScreen("networks");
	}
}

document.addEventListener("DOMContentLoaded", init);

window.addEventListener("beforeunload", () => {
	if (typeof rfidUnsubscribe === "function") {
		rfidUnsubscribe();
		rfidUnsubscribe = null;
	}
});

function updateMeasurementResult(payload) {
	if (!payload || typeof payload !== "object") {
		resetMeasurementResult();
		return;
	}

	const measurementDate = payload.measurementDate || payload.createdAt;
	const containerNumber = payload.container?.number ?? "N/D";
	const zoneName = payload.zone?.name ?? "N/D";
	const contentWeight = typeof payload.contentWeight === "number" ? `${payload.contentWeight}` : payload.contentWeight || "N/D";

	if (resultDateEl) {
		if (measurementDate) {
			const date = new Date(measurementDate);
			const formatted = date.toLocaleTimeString(undefined, {
				hour: "2-digit",
				minute: "2-digit",
				second: "2-digit",
				hour12: false,
			});
			resultDateEl.textContent = formatted;
		} else {
			resultDateEl.textContent = "N/D";
		}
	}

	if (resultContainerEl) {
		resultContainerEl.textContent = containerNumber;
	}

	if (resultZoneEl) {
		resultZoneEl.textContent = zoneName;
	}

	if (resultWeightEl) {
		resultWeightEl.textContent = `${contentWeight} kg`;
	}

	measureDisplay?.classList.add("measure-display--compact");
	measureDetails?.classList.remove("hidden");
}

async function safeJson(response) {
	try {
		return await response.json();
	} catch (_) {
		return null;
	}
}

function resetMeasurementResult() {
	measureDisplay?.classList.remove("measure-display--compact");
	measureDetails?.classList.add("hidden");
	if (resultDateEl) {
		resultDateEl.textContent = "--";
	}
	if (resultContainerEl) {
		resultContainerEl.textContent = "--";
	}
	if (resultZoneEl) {
		resultZoneEl.textContent = "--";
	}
	if (resultWeightEl) {
		resultWeightEl.textContent = "--";
	}
}
