const { app, BrowserWindow, ipcMain, screen, shell } = require('electron');
const path = require('path');
const os = require('os');
const fs = require('fs');
const wifi = require('node-wifi');
const { fork } = require('child_process');

// Inicializa node-wifi usando la interfaz por defecto
wifi.init({ iface: null });

const isWindows = process.platform === 'win32';

function shouldDisableGpu() {
  if (process.argv.includes('--disable-gpu')) {
    return 'CLI flag --disable-gpu';
  }
  if (process.env.DISABLE_GPU === '1') {
    return 'DISABLE_GPU=1';
  }
  if (process.env.DISABLE_GPU === '0') {
    return null;
  }

  const isLinux = process.platform === 'linux';
  const isArm = process.arch.startsWith('arm');
  const runningHeadless = isLinux && !process.env.DISPLAY;

  if (isLinux && (isArm || runningHeadless)) {
    return 'Linux device sin GPU fiable (auto-detectado)';
  }

  return null;
}

const disableGpuReason = shouldDisableGpu();
if (disableGpuReason) {
  console.log(`[start] Deshabilitando aceleración GPU: ${disableGpuReason}`);
  app.disableHardwareAcceleration();
}

// Config file path for persistent settings
const CONFIG_FILE = path.join(app.getPath('userData'), 'scale-config.json');
const SETUP_LOG_FILE = path.join(app.getPath('userData'), 'scale-setup.log');

function appendSetupLogLine(line) {
  try {
    fs.appendFileSync(SETUP_LOG_FILE, `${line}\n`, 'utf8');
  } catch (error) {
    console.error('[Setup log] No se pudo escribir log:', error);
  }
}

function loadScaleConfig() {
  try {
    if (fs.existsSync(CONFIG_FILE)) {
      const data = fs.readFileSync(CONFIG_FILE, 'utf8');
      return JSON.parse(data);
    }
  } catch (error) {
    console.error('[Config] Error loading scale config:', error);
  }
  return {};
}

function saveScaleConfig(config) {
  try {
    fs.writeFileSync(CONFIG_FILE, JSON.stringify(config, null, 2), 'utf8');
    return true;
  } catch (error) {
    console.error('[Config] Error saving scale config:', error);
    return false;
  }
}

function resolvePositiveInt(rawValue, fallback) {
  const parsed = Number.parseInt(rawValue, 10);
  if (Number.isFinite(parsed) && parsed > 0) {
    return parsed;
  }
  return fallback;
}

function normalizeSerialPath(rawPath) {
  if (!rawPath || typeof rawPath !== 'string') {
    return '';
  }

  const trimmed = rawPath.trim();
  if (!trimmed) {
    return '';
  }

  const byIdPrefix = '/dev/serial/by-id/';
  if (!trimmed.startsWith(byIdPrefix)) {
    return trimmed;
  }

  let alias = trimmed.slice(byIdPrefix.length).trim();
  if (!alias) {
    return '';
  }

  if (alias.includes('->')) {
    alias = alias.split('->')[0].trim();
  }

  if (/\s/.test(alias)) {
    const parts = alias.split(/\s+/).filter(Boolean);
    alias = parts[parts.length - 1] || alias;
  }

  return `${byIdPrefix}${alias}`;
}

// Load saved scale config
const savedScaleConfig = loadScaleConfig();
const savedPortNormalized = normalizeSerialPath(savedScaleConfig.port || '');
const savedRfidPortNormalized = normalizeSerialPath(savedScaleConfig.rfidPort || '');
if (savedPortNormalized !== (savedScaleConfig.port || '') || savedRfidPortNormalized !== (savedScaleConfig.rfidPort || '')) {
  savedScaleConfig.port = savedPortNormalized;
  savedScaleConfig.rfidPort = savedRfidPortNormalized;
  saveScaleConfig(savedScaleConfig);
}

const RFID_CONFIG = {
  path:
    (savedScaleConfig && savedScaleConfig.rfidPort) ||
    process.env.RFID_PORT ||
    (isWindows ? 'COM4' : ''),
  baudRate: resolvePositiveInt(process.env.RFID_BAUD_RATE, 115200),
  interval: Math.max(resolvePositiveInt(process.env.RFID_INTERVAL, 500), 250),
  reconnectInterval: Math.max(
    resolvePositiveInt(process.env.RFID_RECONNECT_INTERVAL, 3000),
    1000
  ),
  debug: process.env.RFID_DEBUG === '1',
  execPath: process.env.RFID_NODE_PATH || 'node',
};

const SCALE_CONFIG = {
  port: savedScaleConfig.port || process.env.SCALE_PORT || '',
  baudRate: resolvePositiveInt(savedScaleConfig.baudRate || process.env.SCALE_BAUD, 9600),
  slaveId: resolvePositiveInt(savedScaleConfig.slaveId || process.env.SCALE_SLAVE_ID, 1),
  startAddr: resolvePositiveInt(savedScaleConfig.startAddr || process.env.SCALE_START_ADDR, 0),
  pollInterval: Math.max(resolvePositiveInt(savedScaleConfig.pollInterval || process.env.SCALE_POLL_MS, 500), 100),
  wordOrder: (savedScaleConfig.wordOrder || process.env.SCALE_WORD_ORDER || 'BA').toUpperCase(),
  registers: (savedScaleConfig.registers || process.env.SCALE_REGISTERS || 'holding').toLowerCase(),
  registerCount: Math.max(
    resolvePositiveInt(savedScaleConfig.registerCount || process.env.SCALE_REGISTER_COUNT, 3),
    1
  ),
  divisor: parseFloat(savedScaleConfig.divisor || process.env.SCALE_DIV || '1000'),
  debug: process.env.SCALE_DEBUG === '1',
  execPath: process.env.SCALE_NODE_PATH || 'node',
};

if (!Number.isFinite(SCALE_CONFIG.divisor) || SCALE_CONFIG.divisor <= 0) {
  SCALE_CONFIG.divisor = 1000;
}

/** @type {BrowserWindow | null} */
let mainWindow = null;
let rfidWorker = null;
let rfidReconnectTimer = null;
let scaleWorker = null;
let scaleReconnectTimer = null;
let isAppQuitting = false;
let scaleRequestSeq = 0;
const pendingScaleRequests = new Map();

const rfidState = {
  ready: null,
  tag: null,
  updatedAt: null,
  error: null,
  port: RFID_CONFIG.path,
};

const scaleState = {
  ready: null,
  weight: null,
  rawWeight: null,
  updatedAt: null,
  error: null,
  port: SCALE_CONFIG.port,
};

function hasSerialPortConflict() {
  const rfidPort = (RFID_CONFIG.path || '').trim();
  const scalePort = (SCALE_CONFIG.port || '').trim();
  if (!rfidPort || !scalePort) {
    return false;
  }
  return rfidPort === scalePort;
}

function listStableSerialPorts() {
  return new Promise((resolve) => {
    if (process.platform !== 'linux') {
      resolve({ ok: true, ports: [], source: 'unsupported-platform' });
      return;
    }

    const byIdDir = '/dev/serial/by-id';
    if (!fs.existsSync(byIdDir)) {
      resolve({ ok: true, ports: [], source: 'by-id-empty' });
      return;
    }

    try {
      const aliases = fs.readdirSync(byIdDir).filter(Boolean).sort();
      const ports = [];

      for (const alias of aliases) {
        const fullPath = path.join(byIdDir, alias);
        let target = null;
        try {
          target = fs.readlinkSync(fullPath);
        } catch (_) {
          continue;
        }
        ports.push({
          path: normalizeSerialPath(fullPath),
          alias,
          target,
          label: `${alias} (${target})`,
        });
      }

      resolve({ ok: true, ports, source: 'by-id' });
    } catch (error) {
      resolve({ ok: false, error: error.message || String(error), ports: [] });
    }
  });
}

function hasValidAddress(address) {
  if (!address || typeof address !== 'string') {
    return false;
  }
  if (address === '0.0.0.0' || address === '127.0.0.1') {
    return false;
  }
  if (address.startsWith('169.254.')) {
    return false;
  }
  return true;
}

function detectNetworkConnectivity() {
  const interfaces = os.networkInterfaces();
  const fallback = { hasNetwork: false };

  if (!interfaces || typeof interfaces !== 'object') {
    return fallback;
  }

  for (const [name, infos] of Object.entries(interfaces)) {
    if (!Array.isArray(infos)) {
      continue;
    }

    const validInfo = infos.find(
      (info) =>
        info &&
        info.family === 'IPv4' &&
        hasValidAddress(info.address) &&
        !info.internal
    );

    if (!validInfo) {
      continue;
    }

    const normalizedName = name.toLowerCase();
    const isWifiInterface =
      normalizedName.includes('wi-fi') ||
      normalizedName.includes('wifi') ||
      normalizedName.includes('wlan') ||
      normalizedName.includes('wireless');

    return {
      hasNetwork: true,
      interfaceName: name,
      type: isWifiInterface ? 'wifi' : 'ethernet',
    };
  }

  return fallback;
}

function notifyRfidUpdate() {
  if (!mainWindow || mainWindow.isDestroyed()) {
    return;
  }
  mainWindow.webContents.send('rfid:update', { ...rfidState });
}

function notifyScaleUpdate() {
  if (!mainWindow || mainWindow.isDestroyed()) {
    return;
  }
  mainWindow.webContents.send('scale:update', { ...scaleState });
}

function updateScaleState(patch) {
  if (!patch || typeof patch !== 'object') {
    return;
  }

  let changed = false;
  for (const [key, value] of Object.entries(patch)) {
    if (!(key in scaleState)) {
      continue;
    }
    if (!Object.is(scaleState[key], value)) {
      scaleState[key] = value;
      changed = true;
    }
  }

  if (changed) {
    notifyScaleUpdate();
  }
}

function scheduleScaleReconnect() {
  if (isAppQuitting) {
    return;
  }
  if (scaleReconnectTimer) {
    return;
  }
  scaleReconnectTimer = setTimeout(() => {
    scaleReconnectTimer = null;
    startScaleReader();
  }, 3000);
}

function clearScaleTimers() {
  if (scaleReconnectTimer) {
    clearTimeout(scaleReconnectTimer);
    scaleReconnectTimer = null;
  }
}

function handleScaleWorkerMessage(message) {
  if (!message || typeof message !== 'object') {
    return;
  }

  if (message.type === 'calibrationResult') {
    const requestId = message.requestId;
    if (requestId && pendingScaleRequests.has(requestId)) {
      const entry = pendingScaleRequests.get(requestId);
      pendingScaleRequests.delete(requestId);
      clearTimeout(entry.timeoutId);
      entry.resolve(message);
    }
    return;
  }

  if (message.type === 'clearTareResult') {
    const requestId = message.requestId;
    if (requestId && pendingScaleRequests.has(requestId)) {
      const entry = pendingScaleRequests.get(requestId);
      pendingScaleRequests.delete(requestId);
      clearTimeout(entry.timeoutId);
      entry.resolve(message);
    }
    return;
  }

  if (message.type === 'setupResult') {
    const requestId = message.requestId;
    if (requestId && pendingScaleRequests.has(requestId)) {
      const entry = pendingScaleRequests.get(requestId);
      pendingScaleRequests.delete(requestId);
      clearTimeout(entry.timeoutId);
      entry.resolve(message);
    }
    return;
  }

  if (message.type === 'setupLog') {
    const payload = {
      message: message.message ? String(message.message) : '[setup] (sin mensaje)',
      meta: message.meta && typeof message.meta === 'object' ? message.meta : null,
      timestamp: Number.isFinite(message.timestamp) ? message.timestamp : Date.now(),
    };
    console.log('[Scale setup]', payload.message, payload.meta || '');
    const iso = new Date(payload.timestamp).toISOString();
    appendSetupLogLine(`[${iso}] ${payload.message}${payload.meta ? ` ${JSON.stringify(payload.meta)}` : ''}`);
    if (mainWindow && !mainWindow.isDestroyed()) {
      mainWindow.webContents.send('scale:setup-log', payload);
    }
    return;
  }

  if (message.type === 'update' && message.payload) {
    updateScaleState(message.payload);
    return;
  }

  if (message.type === 'log' && SCALE_CONFIG.debug && message.message) {
    console.log('[Scale worker]', message.message);
    return;
  }

  if (message.type === 'error') {
    const errorText = message.error ? String(message.error) : 'Error en báscula';
    if (SCALE_CONFIG.debug) {
      console.error('[Scale worker] Error:', errorText);
    }
    updateScaleState({ ready: false, error: errorText });
  }
}

function sendScaleWorkerRequest(request, timeoutMs = 60000) {
  if (!scaleWorker) {
    return Promise.resolve({ ok: false, error: 'Servicio de báscula no iniciado' });
  }

  scaleRequestSeq += 1;
  const requestId = `req_${Date.now()}_${scaleRequestSeq}`;

  return new Promise((resolve) => {
    const requestTag = request?.type || 'unknown';
    console.log(`[Scale IPC] request start: ${requestTag} timeout=${timeoutMs}ms`);
    appendSetupLogLine(`[${new Date().toISOString()}] [IPC] start ${requestTag} timeout=${timeoutMs}ms`);
    const timeoutId = setTimeout(() => {
      pendingScaleRequests.delete(requestId);
      console.error(`[Scale IPC] timeout: ${requestTag}`);
      appendSetupLogLine(`[${new Date().toISOString()}] [IPC] timeout ${requestTag}`);
      resolve({
        ok: false,
        error: `Timed out: ${requestTag} superó ${Math.round(timeoutMs / 1000)}s`,
      });
    }, timeoutMs);

    pendingScaleRequests.set(requestId, { resolve, timeoutId });

    try {
      scaleWorker.send({ ...request, requestId });
    } catch (error) {
      clearTimeout(timeoutId);
      pendingScaleRequests.delete(requestId);
      console.error(`[Scale IPC] send error: ${requestTag}`, error);
      resolve({ ok: false, error: error.message || String(error) });
    }
  });
}

function startScaleReader() {
  if (isAppQuitting) {
    return;
  }
  if (scaleWorker) {
    return;
  }

  try {
    scaleWorker = fork(path.join(__dirname, 'scaleWorker.js'), [], {
      stdio: ['inherit', 'inherit', 'inherit', 'ipc'],
      execPath: SCALE_CONFIG.execPath,
      env: {
        ...process.env,
        SCALE_PORT: SCALE_CONFIG.port,
        SCALE_BAUD: String(SCALE_CONFIG.baudRate),
        SCALE_SLAVE_ID: String(SCALE_CONFIG.slaveId),
        SCALE_START_ADDR: String(SCALE_CONFIG.startAddr),
        SCALE_POLL_MS: String(SCALE_CONFIG.pollInterval),
        SCALE_WORD_ORDER: SCALE_CONFIG.wordOrder,
        SCALE_REGISTERS: SCALE_CONFIG.registers,
        SCALE_REGISTER_COUNT: String(SCALE_CONFIG.registerCount),
        SCALE_DIV: String(SCALE_CONFIG.divisor),
        SCALE_DEBUG: SCALE_CONFIG.debug ? '1' : '0',
      },
    });
  } catch (error) {
    console.error('[Scale] No se pudo iniciar el proceso lector:', error);
    updateScaleState({ ready: false, error: error.message });
    scheduleScaleReconnect();
    return;
  }

  updateScaleState({ ready: null, error: null, weight: null, port: SCALE_CONFIG.port });

  scaleWorker.on('message', handleScaleWorkerMessage);
  scaleWorker.on('error', (error) => {
    console.error('[Scale] Error en el proceso lector:', error);
  });
  scaleWorker.on('exit', (code, signal) => {
    if (SCALE_CONFIG.debug) {
      console.log('[Scale] Proceso lector finalizado', { code, signal });
    }
    scaleWorker = null;
    if (!isAppQuitting) {
      updateScaleState({ ready: false });
      scheduleScaleReconnect();
    }
  });
}

function stopScaleReader() {
  clearScaleTimers();
  for (const [requestId, entry] of pendingScaleRequests.entries()) {
    clearTimeout(entry.timeoutId);
    entry.resolve({ ok: false, error: 'Lector de báscula reiniciado' });
    pendingScaleRequests.delete(requestId);
  }
  const worker = scaleWorker;
  scaleWorker = null;
  if (worker) {
    try {
      worker.removeAllListeners();
      worker.kill('SIGTERM');
    } catch (error) {
      console.error('[Scale] No se pudo finalizar el proceso lector:', error);
    }
  }
}

function restartScaleReader() {
  clearScaleTimers();
  const worker = scaleWorker;
  scaleWorker = null;
  let started = false;
  const doStart = () => {
    if (started) return;
    started = true;
    setTimeout(() => startScaleReader(), 400);
  };
  if (worker) {
    worker.removeAllListeners();
    worker.once('exit', doStart);
    try {
      worker.kill('SIGTERM');
    } catch (error) {
      console.error('[Scale] No se pudo finalizar el proceso lector:', error);
      doStart();
    }
    setTimeout(doStart, 5000);
  } else {
    doStart();
  }
}

function updateRfidState(patch) {
  if (!patch || typeof patch !== 'object') {
    return;
  }

  let changed = false;
  for (const [key, value] of Object.entries(patch)) {
    if (!(key in rfidState)) {
      continue;
    }
    if (!Object.is(rfidState[key], value)) {
      rfidState[key] = value;
      changed = true;
    }
  }

  if (changed) {
    notifyRfidUpdate();
  }
}

function scheduleRfidReconnect() {
  if (isAppQuitting) {
    return;
  }
  if (rfidReconnectTimer) {
    return;
  }
  rfidReconnectTimer = setTimeout(() => {
    rfidReconnectTimer = null;
    startRfidReader();
  }, RFID_CONFIG.reconnectInterval);
}

function clearRfidTimers() {
  if (rfidReconnectTimer) {
    clearTimeout(rfidReconnectTimer);
    rfidReconnectTimer = null;
  }
}

function handleRfidWorkerMessage(message) {
  if (!message || typeof message !== 'object') {
    return;
  }

  if (message.type === 'update' && message.payload) {
    updateRfidState(message.payload);
    return;
  }

  if (message.type === 'log' && RFID_CONFIG.debug && message.message) {
    console.log('[RFID worker]', message.message);
    return;
  }

  if (message.type === 'error') {
    const errorText = message.error ? String(message.error) : 'Error en lector RFID';
    if (RFID_CONFIG.debug) {
      console.error('[RFID worker] Error:', errorText);
    }
    updateRfidState({ ready: false, error: errorText });
  }
}

function startRfidReader() {
  if (isAppQuitting) {
    return;
  }
  if (rfidWorker) {
    return;
  }

  if (hasSerialPortConflict()) {
    const message = `RFID y bascula comparten el puerto ${RFID_CONFIG.path}. Ajusta uno de los puertos.`;
    updateRfidState({ ready: false, error: message, port: RFID_CONFIG.path });
    scheduleRfidReconnect();
    return;
  }

  try {
    rfidWorker = fork(path.join(__dirname, 'rfidWorker.js'), [], {
      stdio: ['inherit', 'inherit', 'inherit', 'ipc'],
      execPath: RFID_CONFIG.execPath,
      env: {
        ...process.env,
        RFID_PORT: RFID_CONFIG.path,
        RFID_BAUD_RATE: String(RFID_CONFIG.baudRate),
        RFID_INTERVAL: String(RFID_CONFIG.interval),
        RFID_RECONNECT_INTERVAL: String(RFID_CONFIG.reconnectInterval),
        RFID_DEBUG: RFID_CONFIG.debug ? '1' : '0',
      },
    });
  } catch (error) {
    console.error('[RFID] No se pudo iniciar el proceso lector:', error);
    updateRfidState({ ready: false, error: error.message });
    scheduleRfidReconnect();
    return;
  }

  updateRfidState({ ready: null, error: null, tag: null, port: RFID_CONFIG.path });

  rfidWorker.on('message', handleRfidWorkerMessage);
  rfidWorker.on('error', (error) => {
    console.error('[RFID] Error en el proceso lector:', error);
  });
  rfidWorker.on('exit', (code, signal) => {
    if (RFID_CONFIG.debug) {
      console.log('[RFID] Proceso lector finalizado', { code, signal });
    }
    rfidWorker = null;
    if (!isAppQuitting) {
      updateRfidState({ ready: false });
      scheduleRfidReconnect();
    }
  });
}

function stopRfidReader() {
  clearRfidTimers();
  const worker = rfidWorker;
  rfidWorker = null;
  if (worker) {
    try {
      worker.removeAllListeners();
      worker.kill();
    } catch (error) {
      console.error('[RFID] No se pudo finalizar el proceso lector:', error);
    }
  }
}

function createMainWindow() {
  // Configuración fija para pantalla 640x480
  const windowWidth = 640;
  const windowHeight = 480;

  mainWindow = new BrowserWindow({
    width: windowWidth,
    height: windowHeight,
    resizable: false,
    fullscreen: true,
    kiosk: false,
    autoHideMenuBar: true,
    backgroundColor: '#101419',
    webPreferences: {
      preload: path.join(__dirname, 'preload.js'),
      nodeIntegration: false,
      contextIsolation: true,
      sandbox: false
    }
  });

  mainWindow.loadFile(path.join(__dirname, '../public/index.html'));

  mainWindow.on('closed', () => {
    mainWindow = null;
  });

  mainWindow.webContents.on('did-finish-load', () => {
    notifyRfidUpdate();
notifyScaleUpdate();
  });
}

app.whenReady().then(() => {
  startRfidReader();
  startScaleReader();
  createMainWindow();
});

app.on('activate', () => {
  if (BrowserWindow.getAllWindows().length === 0) {
    createMainWindow();
  }
});

app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') {
    app.quit();
  }
});

app.on('before-quit', () => {
  isAppQuitting = true;
  stopRfidReader();
  stopScaleReader();
});

ipcMain.handle('wifi:get-status', async () => {
  try {
    const connections = await wifi.getCurrentConnections();
    const first = connections?.find((conn) => conn?.ssid);
    const networkStatus = detectNetworkConnectivity();

    if (first && first.ssid) {
      return {
        connected: true,
        ssid: first.ssid,
        hasNetwork: true,
        type: 'wifi',
        interfaceName: networkStatus.interfaceName ?? null,
      };
    }

    return {
      connected: false,
      hasNetwork: Boolean(networkStatus.hasNetwork),
      type: networkStatus.type ?? null,
      interfaceName: networkStatus.interfaceName ?? null,
    };
  } catch (error) {
    const networkStatus = detectNetworkConnectivity();
    return {
      connected: false,
      hasNetwork: Boolean(networkStatus.hasNetwork),
      type: networkStatus.type ?? null,
      interfaceName: networkStatus.interfaceName ?? null,
      error: error.message || String(error),
    };
  }
});

ipcMain.handle('rfid:get-status', async () => {
  return { ...rfidState };
});

ipcMain.handle('wifi:scan', async () => {
  try {
    const networks = await wifi.scan();
    const sorted = networks
      .filter((net) => net.ssid)
      .sort((a, b) => (b.signal_level || 0) - (a.signal_level || 0));
    return { ok: true, networks: sorted };
  } catch (error) {
    return { ok: false, error: error.message || String(error) };
  }
});

ipcMain.handle('wifi:connect', async (_event, payload) => {
  const { ssid, password } = payload || {};
  if (!ssid || typeof ssid !== 'string') {
    return { ok: false, error: 'SSID invalido' };
  }
  try {
    await wifi.connect({ ssid, password: password ?? undefined });
    return { ok: true };
  } catch (error) {
    return { ok: false, error: error.message || String(error) };
  }
});

// Scale IPC handlers
ipcMain.handle('scale:get-status', async () => {
  return { ...scaleState };
});

ipcMain.handle('scale:get-config', async () => {
  return {
    port: SCALE_CONFIG.port,
    baudRate: SCALE_CONFIG.baudRate,
    slaveId: SCALE_CONFIG.slaveId,
    startAddr: SCALE_CONFIG.startAddr,
    pollInterval: SCALE_CONFIG.pollInterval,
    wordOrder: SCALE_CONFIG.wordOrder,
    registers: SCALE_CONFIG.registers,
    divisor: SCALE_CONFIG.divisor,
    registerCount: SCALE_CONFIG.registerCount,
    rfidPort: RFID_CONFIG.path,
  };
});

ipcMain.handle('scale:set-config', async (_event, payload) => {
  if (!payload || typeof payload !== 'object') {
    return { ok: false, error: 'Configuración inválida' };
  }

  const previousRfidPort = RFID_CONFIG.path;

  // Update config
  if (payload.port && typeof payload.port === 'string') {
    const trimmed = normalizeSerialPath(payload.port);
    if (trimmed) {
      SCALE_CONFIG.port = trimmed;
      savedScaleConfig.port = trimmed;
    }
  }
  if (payload.baudRate && Number.isFinite(payload.baudRate)) {
    SCALE_CONFIG.baudRate = payload.baudRate;
  }
  if (payload.slaveId && Number.isFinite(payload.slaveId)) {
    SCALE_CONFIG.slaveId = payload.slaveId;
  }
  if (payload.startAddr !== undefined && Number.isFinite(payload.startAddr)) {
    SCALE_CONFIG.startAddr = payload.startAddr;
  }
  if (payload.pollInterval && Number.isFinite(payload.pollInterval)) {
    SCALE_CONFIG.pollInterval = Math.max(payload.pollInterval, 100);
  }
  if (payload.wordOrder && typeof payload.wordOrder === 'string') {
    SCALE_CONFIG.wordOrder = payload.wordOrder.toUpperCase();
  }
  if (payload.registers && typeof payload.registers === 'string') {
    SCALE_CONFIG.registers = payload.registers.toLowerCase();
  }
  if (payload.divisor && Number.isFinite(payload.divisor)) {
    SCALE_CONFIG.divisor = payload.divisor;
  }
  if (payload.registerCount && Number.isFinite(payload.registerCount)) {
    SCALE_CONFIG.registerCount = Math.max(payload.registerCount, 1);
  }
  if (payload.rfidPort && typeof payload.rfidPort === 'string') {
    const trimmed = normalizeSerialPath(payload.rfidPort);
    if (trimmed) {
      RFID_CONFIG.path = trimmed;
      savedScaleConfig.rfidPort = trimmed;
    }
  }

  // Save to file
  const saved = saveScaleConfig({
    port: SCALE_CONFIG.port,
    baudRate: SCALE_CONFIG.baudRate,
    slaveId: SCALE_CONFIG.slaveId,
    startAddr: SCALE_CONFIG.startAddr,
    pollInterval: SCALE_CONFIG.pollInterval,
    wordOrder: SCALE_CONFIG.wordOrder,
    registers: SCALE_CONFIG.registers,
    divisor: SCALE_CONFIG.divisor,
    registerCount: SCALE_CONFIG.registerCount,
    rfidPort: RFID_CONFIG.path,
  });

  const hasConflict = hasSerialPortConflict();
  const rfidPortChanged = RFID_CONFIG.path !== previousRfidPort;
  if (hasConflict || rfidPortChanged) {
    stopRfidReader();
  }

  // Restart scale reader with new config
  restartScaleReader();

  startRfidReader();

  return { ok: true, saved };
});

ipcMain.handle('scale:calibrate', async () => {
  if (!scaleWorker) {
    return { ok: false, error: 'Servicio de báscula no iniciado' };
  }
  
  if (!scaleState.port) {
      return { ok: false, error: 'Báscula no configurada' };
  }

  // Send message to worker
  scaleWorker.send({ type: 'calibrate' });
  
  // We don't wait for specific response from worker for now, 
  // as the worker will log errors or just continue updating weight which will eventually be 0.
  // But to be nicer we might want to return success immediately if message sent.
  return { ok: true };
});

ipcMain.handle('scale:calibrate-known-weight', async (_event, payload) => {
  const knownWeightKg = Number(payload?.knownWeightKg);
  if (!Number.isFinite(knownWeightKg) || knownWeightKg <= 0) {
    return { ok: false, error: 'Peso patrón inválido' };
  }

  const response = await sendScaleWorkerRequest(
    { type: 'calibrate-known-weight', knownWeightKg },
    60000
  );
  return response;
});

ipcMain.handle('scale:clear-tare', async () => {
  const response = await sendScaleWorkerRequest({ type: 'clear-tare' }, 60000);
  return response;
});

ipcMain.handle('scale:setup-init', async () => {
  const response = await sendScaleWorkerRequest({ type: 'setup-init' }, 60000);
  return response;
});

ipcMain.handle('scale:setup-capture-stable', async () => {
  const response = await sendScaleWorkerRequest({ type: 'setup-capture-stable' }, 60000);
  return response;
});

ipcMain.handle('scale:setup-apply-gain', async (_event, payload) => {
  const knownWeightKg = Number(payload?.knownWeightKg);
  const zeroRaw = Number(payload?.zeroRaw);
  const loadRaw = Number(payload?.loadRaw);
  if (!Number.isFinite(knownWeightKg) || knownWeightKg <= 0) {
    return { ok: false, error: 'Peso patrón inválido' };
  }
  if (!Number.isFinite(zeroRaw) || !Number.isFinite(loadRaw)) {
    return { ok: false, error: 'Lecturas inválidas para ajustar ganancia' };
  }
  const response = await sendScaleWorkerRequest(
    { type: 'setup-apply-gain', knownWeightKg, zeroRaw, loadRaw },
    60000
  );
  return response;
});

ipcMain.handle('scale:setup-save-dead-weight', async () => {
  const response = await sendScaleWorkerRequest({ type: 'setup-save-dead-weight' }, 60000);
  return response;
});

ipcMain.handle('scale:setup-clear-offsets', async (_event, payload) => {
  const preserveDeadWeight = Boolean(payload?.preserveDeadWeight);
  const response = await sendScaleWorkerRequest(
    { type: 'setup-clear-offsets', preserveDeadWeight },
    60000
  );
  return response;
});

ipcMain.handle('scale:setup-read-state', async () => {
  const response = await sendScaleWorkerRequest({ type: 'setup-read-state' }, 60000);
  return response;
});

ipcMain.handle('scale:list-ports', async () => {
  try {
    const { SerialPort } = require('serialport');
    const ports = await SerialPort.list();
    return {
      ok: true,
      ports: ports.map((p) => ({
        path: p.path,
        manufacturer: p.manufacturer || null,
        serialNumber: p.serialNumber || null,
        pnpId: p.pnpId || null,
        friendlyName: p.friendlyName || null,
      })),
    };
  } catch (error) {
    return { ok: false, error: error.message || String(error), ports: [] };
  }
});

ipcMain.handle('scale:list-stable-ports', async () => {
  try {
    return await listStableSerialPorts();
  } catch (error) {
    return { ok: false, error: error.message || String(error), ports: [] };
  }
});

ipcMain.handle('scale:get-setup-log-path', async () => {
  return { ok: true, path: SETUP_LOG_FILE };
});

ipcMain.handle('scale:open-setup-log', async () => {
  try {
    if (!fs.existsSync(SETUP_LOG_FILE)) {
      fs.writeFileSync(SETUP_LOG_FILE, '', 'utf8');
    }
    const openResult = await shell.openPath(SETUP_LOG_FILE);
    if (openResult) {
      return { ok: false, error: openResult, path: SETUP_LOG_FILE };
    }
    return { ok: true, path: SETUP_LOG_FILE };
  } catch (error) {
    return { ok: false, error: error.message || String(error), path: SETUP_LOG_FILE };
  }
});

ipcMain.handle('app:restart-with-update', () => {
  const { spawn } = require('child_process');
  const cmd = process.platform === 'win32' ? 'npm.cmd' : 'npm';

  console.log('[App] Restarting with update...');

  const child = spawn(cmd, ['start'], {
    cwd: path.resolve(__dirname, '..'),
    detached: true,
    stdio: 'ignore'
  });

  child.unref();
  app.quit();
});
